<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Like;

use Erlage\Photogram\System;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Models\Post\PostModelHelper;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Models\Post\Like\PostLikeModel;
use Erlage\Photogram\Data\Models\Post\Like\PostLikeFinder;
use Erlage\Photogram\Data\Models\Post\Like\PostLikeBuilder;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;
use Erlage\Photogram\Data\Models\Notification\NotificationEnum;
use Erlage\Photogram\Data\Tables\Notification\NotificationTable;
use Erlage\Photogram\Data\Models\Notification\NotificationFinder;
use Erlage\Photogram\Data\Models\Notification\NotificationBuilder;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowFinder;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowBuilder;
use Erlage\Photogram\Data\Dtos\Notification\NotificationLinkedContentDTO;

final class PostLikeActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | do like
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::postlikeProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | undo like
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::postlikeProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | processor
    |--------------------------------------------------------------------------
    */

    private static function postlikeProcessor(bool $doLike): void
    {
        self::process(function () use ($doLike)
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateCache = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(
                PostLikeTable::LIKED_POST_ID,
                RequestTable::PAYLOAD,
                PostLikeTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | if undo like request
            |--------------------------------------------------------------------------
            */

            if (false == $doLike)
            {
                $finder = (new PostLikeFinder())
                    -> setLikedPostId($targetPostModel -> getId())
                    -> setLikedByUserId(self::$authedUserModel -> getId())
                    -> find();

                if ($finder -> isFound())
                {
                    $finder -> popModelFromResults() -> delete();

                    $flagUpdateCache = true;
                }
            }
            /*
            |--------------------------------------------------------------------------
            | if it's a do like request
            |--------------------------------------------------------------------------
            */
            elseif (true == $doLike)
            {
                /*
                |--------------------------------------------------------------------------
                | try selecting the post like first
                |--------------------------------------------------------------------------
                */

                $postLikeBean = (new Query())
                    -> from(PostLikeTable::TABLE_NAME)
                    -> where(PostLikeTable::LIKED_POST_ID, $targetPostModel -> getId())
                    -> where(PostLikeTable::LIKED_BY_USER_ID, self::$authedUserModel -> getId())
                    -> selectOne();

                /*
                |--------------------------------------------------------------------------
                | if post is already liked
                |--------------------------------------------------------------------------
                */

                if (null != $postLikeBean)
                {
                    /*
                    |--------------------------------------------------------------------------
                    | create post like model from selected bean
                    |--------------------------------------------------------------------------
                    */

                    $postLikeModel = PostLikeModel::createFromUntouchedBean_noException($postLikeBean);

                    self::ensureModel($postLikeModel);

                    /*
                    |--------------------------------------------------------------------------
                    | add to post like model to post likes map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(PostLikeTable::getTableName(), $postLikeModel -> getDataMap());
                }

                /*
                |--------------------------------------------------------------------------
                | else if post is not already liked
                |--------------------------------------------------------------------------
                */

                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | build post like
                    |--------------------------------------------------------------------------
                    */

                    $postLikeModel = (new PostLikeBuilder())
                        -> setLikedPostId($targetPostModel -> getId())
                        -> setLikedByUserId(self::$authedUserModel -> getId())
                        -> dispense();

                    /*
                    |--------------------------------------------------------------------------
                    | do post like
                    |--------------------------------------------------------------------------
                    */

                    $postLikeModel -> save();

                    /*
                    |--------------------------------------------------------------------------
                    |  if everything is alright, add postlike to postlikes map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(PostLikeTable::getTableName(), $postLikeModel -> getDataMap());

                    $flagUpdateCache = true;
                }
            }

            if ($flagUpdateCache)
            {
                PostModelHelper::updateCacheLikesCount($targetPostModel);

                /*
                |--------------------------------------------------------------------------
                | send "liked post" notification
                |--------------------------------------------------------------------------
                */

                if ($doLike)
                {
                    $notifyPostOwner = false;

                    /*
                    |--------------------------------------------------------------------------
                    | make sure we're not sending notifications to ourself
                    |--------------------------------------------------------------------------
                    */
                    if ($postOwnerUserModel -> getId() != self::$authedUserModel -> getId())
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | check if target user has notifications enabled
                        |--------------------------------------------------------------------------
                        */

                        $metaPushSettings = $postOwnerUserModel -> getMetaPushSettings();

                        // if notifications are on for post likes
                        if (UserMetaPushSettingsDTO::VALUE_OFF != $metaPushSettings -> getLikes())
                        {
                            // if notifications are on for everyone
                            if (UserMetaPushSettingsDTO::VALUE_FROM_EVERYONE == $metaPushSettings -> getLikes())
                            {
                                $notifyPostOwner = true;
                            }
                            // else if only from people they follow
                            elseif (UserMetaPushSettingsDTO::VALUE_FROM_PEOPLE_I_FOLLOW == $metaPushSettings -> getLikes())
                            {
                                // check whether user is followed
                                $notifyPostOwner = CommonQueries::isFollowing(
                                    $postOwnerUserModel -> getId(),
                                    self::$authedUserModel -> getId()
                                );
                            }
                        }
                    }

                    if ($notifyPostOwner)
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | first try to find if someone already liked post today
                        |--------------------------------------------------------------------------
                        */

                        $finder = (new NotificationFinder())
                            -> setToUserId($postOwnerUserModel -> getId())
                            -> setTargetContentId($targetPostModel -> getId())
                            -> setMetaType(NotificationEnum::META_TYPE_POST_LIKE)
                            -> setStampRegistrationDate(System::isoDate())
                            -> find();

                        if ($finder -> isFound())
                        {
                            /*
                            |--------------------------------------------------------------------------
                            | notification exists, simply add user to existing notifcation
                            |--------------------------------------------------------------------------
                            */

                            $notifcationModel = $finder -> popModelFromResults();

                            $notificationContent = $notifcationModel -> getLinkedContent();

                            if ( ! $notificationContent -> getUserIds() -> containsValue(self::$authedUserModel -> getId()))
                            {
                                $notificationContent -> linkUserId(self::$authedUserModel -> getId());

                                $notifcationModel -> update(
                                    array(
                                        NotificationTable::LINKED_CONTENT => $notificationContent,
                                        NotificationTable::META_IS_READ   => NotificationEnum::META_IS_READ_NO,
                                    )
                                );

                                $notifcationModel -> save();
                            }
                        }
                        else
                        {
                            /*
                            |--------------------------------------------------------------------------
                            | notification doesnt exists, create and dispatch a new notification
                            |--------------------------------------------------------------------------
                            */

                            $notificationContent = new NotificationLinkedContentDTO();

                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                            $notificationContent -> linkPostId($targetPostModel -> getId());

                            $notificationModel = (new NotificationBuilder())
                                -> setToUserId($postOwnerUserModel -> getId())
                                -> setTargetContentId($targetPostModel -> getId())
                                -> setMetaType(NotificationEnum::META_TYPE_POST_LIKE)
                                -> setLinkedContent($notificationContent)
                                -> dispense();

                            $notificationModel -> save();
                        }
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | notify users who are tagged in the post which just got liked
                |--------------------------------------------------------------------------
                */

                if ($doLike)
                {
                    $targetPostDisplayContent = $targetPostModel -> getDisplayContent();

                    $targetPostDisplayContent -> iteratorPrepare();

                    /*
                    |--------------------------------------------------------------------------
                    | for all images in post content
                    |--------------------------------------------------------------------------
                    */

                    while ($targetPostDisplayContent -> iteratorForward())
                    {
                        $displayContentItemDTO = $targetPostDisplayContent -> iteratorCurrentDisplayContentItem();

                        /*
                        |--------------------------------------------------------------------------
                        | get tagged users DTO
                        |--------------------------------------------------------------------------
                        */

                        $taggedUsersDTO = $displayContentItemDTO -> getDisplayUserTagsOnItemDTO();

                        /*
                        |--------------------------------------------------------------------------
                        | for all tagged users
                        |--------------------------------------------------------------------------
                        */

                        $taggedUsersDTO -> iteratorPrepare();

                        while ($taggedUsersDTO -> iteratorForward())
                        {
                            $taggedUserDTO = $taggedUsersDTO -> iteratorCurrentDisplayUserTag();

                            /*
                            |--------------------------------------------------------------------------
                            | if tagged user is post owner then don't send them a notification
                            |--------------------------------------------------------------------------
                            */

                            if ($postOwnerUserModel -> getId() == $taggedUserDTO -> getUserId())
                            {
                                continue;
                            }

                            /*
                            |--------------------------------------------------------------------------
                            | get tagged user model
                            |--------------------------------------------------------------------------
                            */

                            $taggedUserModel = UserModel::findFromId_noException($taggedUserDTO -> getUserId());

                            if ($taggedUserModel -> isModel())
                            {
                                $pushSettings = $taggedUserModel -> getMetaPushSettings();

                                /*
                                |--------------------------------------------------------------------------
                                | if tagged user has opted in for notifications
                                |--------------------------------------------------------------------------
                                */

                                if (UserMetaPushSettingsDTO::VALUE_ON == $pushSettings -> getLikesOnPhotosOfYou())
                                {
                                    /*
                                    |--------------------------------------------------------------------------
                                    | first try to find if someone already liked post today and notif is sent
                                    |--------------------------------------------------------------------------
                                    */

                                    $finder = (new NotificationFinder())
                                        -> setToUserId($taggedUserModel -> getId())
                                        -> setTargetContentId($targetPostModel -> getId())
                                        -> setMetaType(NotificationEnum::META_TYPE_POST_LIKE_ON_PHOTO_OF_YOU)
                                        -> setStampRegistrationDate(System::isoDate())
                                        -> find();

                                    if ($finder -> isFound())
                                    {
                                        /*
                                        |--------------------------------------------------------------------------
                                        | notification exists, simply add user to existing notifcation
                                        |--------------------------------------------------------------------------
                                        */

                                        $notifcationModel = $finder -> popModelFromResults();

                                        $notificationContent = $notifcationModel -> getLinkedContent();

                                        if ( ! $notificationContent -> getUserIds() -> containsValue(self::$authedUserModel -> getId()))
                                        {
                                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());

                                            $notifcationModel -> update(
                                                array(
                                                    NotificationTable::LINKED_CONTENT => $notificationContent,
                                                    NotificationTable::META_IS_READ   => NotificationEnum::META_IS_READ_NO,
                                                )
                                            );

                                            $notifcationModel -> save();
                                        }
                                    }
                                    else
                                    {
                                        /*
                                        |--------------------------------------------------------------------------
                                        | notification doesnt exists, create and dispatch a new notification
                                        |--------------------------------------------------------------------------
                                        */

                                        $notificationContent = new NotificationLinkedContentDTO();

                                        $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                                        $notificationContent -> linkPostId($targetPostModel -> getId());

                                        $notificationModel = (new NotificationBuilder())
                                            -> setToUserId($taggedUserModel -> getId())
                                            -> setTargetContentId($targetPostModel -> getId())
                                            -> setMetaType(NotificationEnum::META_TYPE_POST_LIKE_ON_PHOTO_OF_YOU)
                                            -> setLinkedContent($notificationContent)
                                            -> dispense();

                                        $notificationModel -> save();
                                    }
                                }
                            }
                        }
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | if a post is liked, tweak user feeds if enabled
                |--------------------------------------------------------------------------
                */

                if ($doLike && Settings::getBool(ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_LIKE))
                {
                    $metaHashtags = $targetPostModel -> getMetaHashtags();

                    $metaHashtags -> iteratorPrepare();

                    while ($metaHashtags -> iteratorForward())
                    {
                        $hashtagId = $metaHashtags -> iteratorCurrentHashtagId();

                        $finder = (new HashtagFollowFinder())
                            -> setFollowedHashtagId($hashtagId)
                            -> setFollowedByUserId(self::$authedUserModel -> getId())
                            -> find();

                        if ($finder -> notFound())
                        {
                            $hashtagFollowModel = (new HashtagFollowBuilder())
                                -> setFollowedHashtagId($hashtagId)
                                -> setFollowedByUserId(self::$authedUserModel -> getId())
                                -> dispense();

                            $hashtagFollowModel -> save();
                        }
                    }
                }
            }
        });
    }
}
